<?php
// اتصال به دیتابیس
require_once __DIR__ . '/../config.php';

// شروع سشن
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

/**
 * بارگذاری زبان
 */
if (!function_exists('load_lang')) {
    function load_lang(): array {
        $lang = get_lang(); // تابع get_lang باید زبان انتخابی رو برگردونه (fa یا en)
        $langFile = __DIR__ . '/../lang/' . $lang . '.php';

        if (file_exists($langFile)) {
            return require $langFile;
        }
        return require __DIR__ . '/../lang/fa.php'; // fallback
    }
}

/**
 * ثبت‌نام کاربر
 */
if (!function_exists('process_register')) {
    function process_register(array $L): array {
        global $pdo;
        $flashes = [];

        $fullname_fa = trim($_POST['fullname_fa'] ?? '');
        $fullname_en = trim($_POST['fullname_en'] ?? '');
        $username    = trim($_POST['username'] ?? '');
        $password    = trim($_POST['password'] ?? '');
        $confirm     = trim($_POST['password_confirm'] ?? '');
        $mobile      = trim($_POST['mobile'] ?? '');
        $email       = trim($_POST['email'] ?? '');
        $verify      = trim($_POST['verify'] ?? '');

        // بررسی مقادیر
        if ($fullname_fa === '' || $fullname_en === '' || $username === '' || $password === '' ||
            $confirm === '' || $mobile === '' || $email === '' || $verify === '') {
            $flashes['error'] = $L['fill_all_fields'] ?? 'لطفاً همه فیلدها را پر کنید.';
            return $flashes;
        }
        if ($password !== $confirm) {
            $flashes['error'] = $L['password_mismatch'] ?? 'رمز عبور و تکرار آن یکسان نیست.';
            return $flashes;
        }
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $flashes['error'] = $L['invalid_email'] ?? 'ایمیل معتبر نیست.';
            return $flashes;
        }
        if (!preg_match('/^[0-9]{4}$/', $verify)) {
            $flashes['error'] = $L['invalid_verify'] ?? 'کد تأیید معتبر نیست.';
            return $flashes;
        }

        // بررسی یکتا بودن کاربر
        $stmt = $pdo->prepare("SELECT id FROM users WHERE username = :username OR email = :email");
        $stmt->execute([':username' => $username, ':email' => $email]);
        if ($stmt->fetch()) {
            $flashes['error'] = $L['user_exists'] ?? 'این نام کاربری یا ایمیل قبلاً ثبت شده است.';
            return $flashes;
        }

        // هش کردن رمز
        $hash = password_hash($password, PASSWORD_DEFAULT);

        // ذخیره در دیتابیس
        $stmt = $pdo->prepare("
            INSERT INTO users (fullname_fa, fullname_en, username, password, mobile, email, verify_code, status)
            VALUES (:fullname_fa, :fullname_en, :username, :password, :mobile, :email, :verify, 'pending')
        ");
        $stmt->execute([
            ':fullname_fa' => $fullname_fa,
            ':fullname_en' => $fullname_en,
            ':username'    => $username,
            ':password'    => $hash,
            ':mobile'      => $mobile,
            ':email'       => $email,
            ':verify'      => $verify,
        ]);

        $flashes['success'] = $L['register_success'] ?? 'ثبت‌نام با موفقیت انجام شد.';
        return $flashes;
    }
}

/**
 * ورود کاربر
 */
if (!function_exists('process_login')) {
    function process_login(array $L): array {
        global $pdo;
        $flashes = [];

        $username = trim($_POST['username'] ?? '');
        $password = trim($_POST['password'] ?? '');

        if ($username === '' || $password === '') {
            $flashes['error'] = $L['fill_all_fields'] ?? 'لطفاً همه فیلدها را پر کنید.';
            return $flashes;
        }

        // پیدا کردن کاربر
        $stmt = $pdo->prepare("SELECT id, password, status 
                               FROM users 
                               WHERE username = :username OR email = :email 
                               LIMIT 1");
        $stmt->execute([
            ':username' => $username,
            ':email'    => $username
        ]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$user || !password_verify($password, $user['password'])) {
            $flashes['error'] = $L['invalid_credentials'] ?? 'نام کاربری یا رمز عبور اشتباه است.';
            return $flashes;
        }

        if ($user['status'] !== 'active') {
            $flashes['error'] = $L['inactive_account'] ?? 'حساب کاربری شما فعال نشده است.';
            return $flashes;
        }

        // ذخیره‌ی سشن
        $_SESSION['user_id'] = $user['id'];
        $flashes['success'] = $L['login_success'] ?? 'با موفقیت وارد شدید.';
        
        // ریدایرکت به داشبورد
        header("Location: ../dashboard.php");
        exit;
        
        return $flashes;
    }
}

// بارگذاری زبان
$L = load_lang();

// اگر درخواست ثبت‌نام ارسال شد
if (isset($_POST['action']) && $_POST['action'] === 'register') {
    $result = process_register($L);

    if (isset($result['error'])) {
        echo $result['error'];
    } elseif (isset($result['success'])) {
        echo $result['success'];
    }
}

// اگر درخواست ورود ارسال شد
if (isset($_POST['action']) && $_POST['action'] === 'login') {
    $result = process_login($L);

    if (isset($result['error'])) {
        echo $result['error'];
    } elseif (isset($result['success'])) {
        echo $result['success'];
    }
}